#include <Windows.h>

class DebugClass
{
public:
 
    DebugClass(const char* Function, const char* File, int Line, bool LogPerformance, const char* Comment)
        : m_startCount(0), m_elapsedTimeSec(0.0)
    {
        //
        // Confine the test to run on a single processor,
        // to get consistent results for all tests.
        //
 //       SetThreadAffinityMask(GetCurrentThread(), 1);
 //       SetThreadIdealProcessor(GetCurrentThread(), 0);

		m_function = Function;
		m_file = File;
		m_line = Line;
		m_logperformance = LogPerformance;
		m_comment = Comment;

        Sleep(1);
		Start();
    }

	~DebugClass()
	{
		Stop(); 
		StringClass Text = StringClass::getFormattedString("%s() %s:%d finished", m_function, m_file, m_line);

		if (m_logperformance)
		{
			Text += StringClass::getFormattedString(" in: %f milliseconds", ElapsedTimeMilliSec());
		}
		if (m_comment != NULL)
		{
			Text += StringClass::getFormattedString(". COMMENT: %s", m_comment);
		}
		Functions::Debug_Log(Text);
	} 
 
    // Starts measuring performance
    // (to be called before the block of code to measure).
    void Start()
    {
        // Clear total elapsed time
        // (it is a spurious value until Stop() is called)
        m_elapsedTimeSec = 0.0;
 
        // Start ticking
        m_startCount = Counter();
    }
 
    // Stops measuring performance
    // (to be called after the block of code to measure).
    void Stop()
    {
        // Stop ticking
        LONGLONG stopCount = Counter();
 
        // Calculate total elapsed time since Start() was called;
        // time is measured in seconds
        m_elapsedTimeSec = (stopCount - m_startCount) * 1.0 / Frequency();
 
        // Clear start count (it is spurious information)
        m_startCount = 0;
    }
 
    // Returns total elapsed time (in seconds) in Start-Stop interval.
    double ElapsedTimeSec() const
    {
        // Total elapsed time was calculated in Stop() method.
        return m_elapsedTimeSec;
    }
 
    // Returns total elapsed time (in milliseconds) in Start-Stop interval.
    double ElapsedTimeMilliSec() const
    {
        // Total elapsed time was calculated in Stop() method.
        return m_elapsedTimeSec * 1000.0;
    }
 
 
 
    //--------------------------------------------------------------------
    // IMPLEMENTATION
    //--------------------------------------------------------------------
private:
 
    //
    // *** Data Members ***
    //
 
    // The value of counter on start ticking
    LONGLONG m_startCount;
 
    // The time (in seconds) elapsed in Start-Stop interval
    double m_elapsedTimeSec;

	const char* m_function;
	const char* m_file;
	const char* m_comment;
	int m_line;
	bool m_logperformance;
 
    //
    // *** Helper Methods ***
    //
 
    // Returns current value of high-resolution performance counter.
    LONGLONG Counter() const
    {
        LARGE_INTEGER counter;
        QueryPerformanceCounter(&counter);
        return counter.QuadPart;
    }
 
    // Returns the frequency (in counts per second) of the
    // high-resolution performance counter.
    LONGLONG Frequency() const
    {
        LARGE_INTEGER frequency;
        QueryPerformanceFrequency(&frequency);
        return frequency.QuadPart;
    }
 
 
    //
    // *** Ban copy ***
    //
private:
    DebugClass(const DebugClass &);
    DebugClass & operator=(const DebugClass &);
};